<?php
/**
 * Survey Response Analysis Tool
 * 
 * This script helps analyze the collected survey responses
 * Run this script to get insights from your survey data
 */

class SurveyAnalyzer {
    private $responses = [];
    private $questions = [];
    
    public function __construct() {
        $this->loadResponses();
        $this->loadQuestions();
    }
    
    private function loadResponses() {
        $responses_dir = 'responses/';
        if (!is_dir($responses_dir)) {
            echo "No responses directory found.\n";
            return;
        }
        
        $files = glob($responses_dir . '*.json');
        foreach ($files as $file) {
            $data = json_decode(file_get_contents($file), true);
            if ($data) {
                $this->responses = array_merge($this->responses, $data);
            }
        }
    }
    
    private function loadQuestions() {
        // Load English questions by default for analysis
        $buyer_questions = json_decode(file_get_contents('buyer_questions.json'), true);
        $seller_questions = json_decode(file_get_contents('seller_questions.json'), true);
        
        $this->questions = [
            'buyer' => $buyer_questions['questions'],
            'seller' => $seller_questions['questions']
        ];
    }
    
    public function getBasicStats() {
        $stats = [
            'total_responses' => count($this->responses),
            'buyer_responses' => 0,
            'seller_responses' => 0,
            'languages' => [],
            'user_paths' => [],
            'dates' => []
        ];
        
        foreach ($this->responses as $response) {
            $stats['buyer_responses'] += ($response['survey_type'] === 'buyer') ? 1 : 0;
            $stats['seller_responses'] += ($response['survey_type'] === 'seller') ? 1 : 0;
            
            $stats['languages'][$response['language']] = ($stats['languages'][$response['language']] ?? 0) + 1;
            $stats['user_paths'][$response['user_path']] = ($stats['user_paths'][$response['user_path']] ?? 0) + 1;
            
            $date = date('Y-m-d', strtotime($response['timestamp']));
            $stats['dates'][$date] = ($stats['dates'][$date] ?? 0) + 1;
        }
        
        return $stats;
    }
    
    public function analyzeQuestion($question_id, $survey_type = 'buyer') {
        $question = $this->findQuestion($question_id, $survey_type);
        if (!$question) {
            return null;
        }
        
        $responses = $this->getResponsesForQuestion($question_id, $survey_type);
        $analysis = [
            'question' => $question['question'],
            'type' => $question['type'],
            'total_answers' => count($responses),
            'answers' => []
        ];
        
        if ($question['type'] === 'single-choice') {
            $analysis['answers'] = $this->analyzeSingleChoice($responses);
        } elseif ($question['type'] === 'multi-choice') {
            $analysis['answers'] = $this->analyzeMultiChoice($responses);
        } elseif ($question['type'] === 'matrix') {
            $analysis['answers'] = $this->analyzeMatrix($responses, $question);
        } elseif ($question['type'] === 'open-text') {
            $analysis['answers'] = $this->analyzeOpenText($responses);
        }
        
        return $analysis;
    }
    
    private function findQuestion($question_id, $survey_type) {
        foreach ($this->questions[$survey_type] as $question) {
            if ($question['id'] == $question_id) {
                return $question;
            }
        }
        return null;
    }
    
    private function getResponsesForQuestion($question_id, $survey_type) {
        $question_responses = [];
        
        foreach ($this->responses as $response) {
            if ($response['survey_type'] === $survey_type && isset($response['responses'])) {
                $responses = is_string($response['responses']) ? json_decode($response['responses'], true) : $response['responses'];
                if (isset($responses[$question_id])) {
                    $question_responses[] = $responses[$question_id];
                }
            }
        }
        
        return $question_responses;
    }
    
    private function analyzeSingleChoice($responses) {
        $counts = [];
        foreach ($responses as $response) {
            $counts[$response] = ($counts[$response] ?? 0) + 1;
        }
        
        $total = count($responses);
        $percentages = [];
        foreach ($counts as $option => $count) {
            $percentages[$option] = [
                'count' => $count,
                'percentage' => round(($count / $total) * 100, 2)
            ];
        }
        
        return $percentages;
    }
    
    private function analyzeMultiChoice($responses) {
        $counts = [];
        $total_responses = count($responses);
        
        foreach ($responses as $response) {
            if (is_array($response)) {
                foreach ($response as $option) {
                    $counts[$option] = ($counts[$option] ?? 0) + 1;
                }
            }
        }
        
        $percentages = [];
        foreach ($counts as $option => $count) {
            $percentages[$option] = [
                'count' => $count,
                'percentage' => round(($count / $total_responses) * 100, 2)
            ];
        }
        
        return $percentages;
    }
    
    private function analyzeMatrix($responses, $question) {
        $analysis = [];
        
        foreach ($question['options'] as $option) {
            $analysis[$option] = [];
            foreach ($question['scale'] as $scale_value) {
                $analysis[$option][$scale_value] = 0;
            }
        }
        
        foreach ($responses as $response) {
            if (is_array($response)) {
                foreach ($response as $option => $value) {
                    if (isset($analysis[$option][$value])) {
                        $analysis[$option][$value]++;
                    }
                }
            }
        }
        
        return $analysis;
    }
    
    private function analyzeOpenText($responses) {
        return [
            'total_responses' => count($responses),
            'sample_responses' => array_slice($responses, 0, 5) // Show first 5 responses
        ];
    }
    
    public function generateReport() {
        $stats = $this->getBasicStats();
        
        echo "=== SURVEY ANALYSIS REPORT ===\n\n";
        echo "Basic Statistics:\n";
        echo "- Total Responses: {$stats['total_responses']}\n";
        echo "- Buyer Responses: {$stats['buyer_responses']}\n";
        echo "- Seller Responses: {$stats['seller_responses']}\n\n";
        
        echo "Language Distribution:\n";
        foreach ($stats['languages'] as $lang => $count) {
            $percentage = round(($count / $stats['total_responses']) * 100, 2);
            echo "- {$lang}: {$count} ({$percentage}%)\n";
        }
        echo "\n";
        
        echo "User Path Distribution:\n";
        foreach ($stats['user_paths'] as $path => $count) {
            $percentage = round(($count / $stats['total_responses']) * 100, 2);
            echo "- {$path}: {$count} ({$percentage}%)\n";
        }
        echo "\n";
        
        echo "Response Timeline:\n";
        ksort($stats['dates']);
        foreach ($stats['dates'] as $date => $count) {
            echo "- {$date}: {$count} responses\n";
        }
        echo "\n";
        
        // Analyze key questions
        if ($stats['buyer_responses'] > 0) {
            echo "=== BUYER SURVEY KEY INSIGHTS ===\n\n";
            
            // Question 1: Have you bought from small businesses?
            $q1_analysis = $this->analyzeQuestion(1, 'buyer');
            if ($q1_analysis) {
                echo "Q1: {$q1_analysis['question']}\n";
                foreach ($q1_analysis['answers'] as $option => $data) {
                    echo "  - {$option}: {$data['count']} ({$data['percentage']}%)\n";
                }
                echo "\n";
            }
            
            // Question 3: How often do you buy?
            $q3_analysis = $this->analyzeQuestion(3, 'buyer');
            if ($q3_analysis) {
                echo "Q3: {$q3_analysis['question']}\n";
                foreach ($q3_analysis['answers'] as $option => $data) {
                    echo "  - {$option}: {$data['count']} ({$data['percentage']}%)\n";
                }
                echo "\n";
            }
        }
        
        if ($stats['seller_responses'] > 0) {
            echo "=== SELLER SURVEY KEY INSIGHTS ===\n\n";
            
            // Question 1: Do you currently sell?
            $q1_analysis = $this->analyzeQuestion(1, 'seller');
            if ($q1_analysis) {
                echo "Q1: {$q1_analysis['question']}\n";
                foreach ($q1_analysis['answers'] as $option => $data) {
                    echo "  - {$option}: {$data['count']} ({$data['percentage']}%)\n";
                }
                echo "\n";
            }
        }
    }
    
    public function exportToCSV($filename = 'survey_analysis.csv') {
        $stats = $this->getBasicStats();
        
        $csv_data = [];
        $csv_data[] = ['Question', 'Option', 'Count', 'Percentage', 'Survey Type'];
        
        // Export buyer survey data
        if ($stats['buyer_responses'] > 0) {
            foreach ($this->questions['buyer'] as $question) {
                $analysis = $this->analyzeQuestion($question['id'], 'buyer');
                if ($analysis && $analysis['type'] !== 'open-text') {
                    foreach ($analysis['answers'] as $option => $data) {
                        $csv_data[] = [
                            $question['question'],
                            $option,
                            $data['count'],
                            $data['percentage'],
                            'buyer'
                        ];
                    }
                }
            }
        }
        
        // Export seller survey data
        if ($stats['seller_responses'] > 0) {
            foreach ($this->questions['seller'] as $question) {
                $analysis = $this->analyzeQuestion($question['id'], 'seller');
                if ($analysis && $analysis['type'] !== 'open-text') {
                    foreach ($analysis['answers'] as $option => $data) {
                        $csv_data[] = [
                            $question['question'],
                            $option,
                            $data['count'],
                            $data['percentage'],
                            'seller'
                        ];
                    }
                }
            }
        }
        
        $file = fopen($filename, 'w');
        foreach ($csv_data as $row) {
            fputcsv($file, $row);
        }
        fclose($file);
        
        echo "Data exported to {$filename}\n";
    }
}

// Usage
if (php_sapi_name() === 'cli') {
    $analyzer = new SurveyAnalyzer();
    $analyzer->generateReport();
    
    // Uncomment the line below to export to CSV
    // $analyzer->exportToCSV();
} else {
    echo "This script is designed to be run from the command line.\n";
    echo "Usage: php analyze_responses.php\n";
}
?>
