<?php
session_start();

$type = $_GET['type'] ?? 'buyer';
$lang = $_GET['lang'] ?? 'en';

// Load questions based on type and language
$base_file = $type === 'seller' ? 'seller_questions' : 'buyer_questions';
$questions_file = $lang === 'bg' ? $base_file . '_bg.json' : $base_file . '.json';
$questions_data = json_decode(file_get_contents($questions_file), true);

// Language translations for UI elements
$translations = [
    'en' => [
        'next' => 'Next',
        'previous' => 'Previous',
        'submit' => 'Submit Survey',
        'question' => 'Question',
        'of' => 'of',
        'required' => 'Required',
        'optional' => 'Optional'
    ],
    'bg' => [
        'next' => 'Напред',
        'previous' => 'Назад',
        'submit' => 'Изпрати проучването',
        'question' => 'Въпрос',
        'of' => 'от',
        'required' => 'Задължително',
        'optional' => 'По желание'
    ]
];

$current_lang = $translations[$lang] ?? $translations['en'];
?>

<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $questions_data['title']; ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="container">
        <div class="survey-container">
            <div class="survey-header">
                <h1 class="survey-title"><?php echo $questions_data['title']; ?></h1>
                <p class="survey-description"><?php echo $questions_data['description']; ?></p>
            </div>
            
            <div class="progress-container">
                <div class="progress-bar">
                    <div class="progress-fill" id="progressFill"></div>
                </div>
                <div class="progress-text" id="progressText"><?php echo $current_lang['question']; ?> 1 <?php echo $current_lang['of']; ?> <?php echo count($questions_data['questions']); ?></div>
            </div>
            
            <!-- Mobile Debug Info -->
            <div style="text-align: center; margin: 10px 0;">
                <button type="button" id="debugToggle" style="background: #ff6b6b; color: white; border: none; padding: 5px 10px; border-radius: 4px; font-size: 12px;">
                    🔧 Show Debug
                </button>
            </div>
            <div id="mobileDebug" style="background: #f0f0f0; padding: 10px; margin: 10px 0; border-radius: 8px; font-size: 12px; display: none;">
                <strong>Debug Info:</strong><br>
                <span id="debugCurrent">Current: 0</span><br>
                <span id="debugTotal">Total: <?php echo count($questions_data['questions']); ?></span><br>
                <span id="debugVisible">Visible Questions: 0</span><br>
                <span id="debugActive">Active Question: None</span><br>
                <span id="debugLastAction">Last Action: None</span>
            </div>
            
            <form id="surveyForm" method="POST" action="survey.php">
                <input type="hidden" name="action" value="save_survey">
                <input type="hidden" name="survey_type" value="<?php echo $type; ?>">
                <input type="hidden" name="lang" value="<?php echo $lang; ?>">
                <input type="hidden" name="responses" id="responsesInput">
                
                <?php foreach ($questions_data['questions'] as $index => $question): ?>
                <div class="question-container <?php echo $index === 0 ? 'active' : ''; ?>" data-question="<?php echo $question['id']; ?>" style="<?php echo $index === 0 ? 'display: block; opacity: 1; transform: translateY(0);' : 'display: none; opacity: 0; transform: translateY(20px);'; ?>">
                    <h2 class="question-text">
                        <?php echo $question['question']; ?>
                        <?php if ($question['type'] === 'open-text'): ?>
                            <span class="optional-text">(<?php echo $current_lang['optional']; ?>)</span>
                        <?php endif; ?>
                    </h2>
                    
                    <?php if ($question['type'] === 'single-choice'): ?>
                        <div class="options-container">
                            <?php foreach ($question['options'] as $option): ?>
                            <div class="option" data-value="<?php echo htmlspecialchars($option); ?>">
                                <input type="radio" name="q<?php echo $question['id']; ?>" value="<?php echo htmlspecialchars($option); ?>" id="q<?php echo $question['id']; ?>_<?php echo array_search($option, $question['options']); ?>" style="display: none;">
                                <label for="q<?php echo $question['id']; ?>_<?php echo array_search($option, $question['options']); ?>"><?php echo htmlspecialchars($option); ?></label>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    <?php elseif ($question['type'] === 'multi-choice'): ?>
                        <div class="options-container">
                            <?php foreach ($question['options'] as $option): ?>
                            <div class="option" data-value="<?php echo htmlspecialchars($option); ?>">
                                <input type="checkbox" name="q<?php echo $question['id']; ?>[]" value="<?php echo htmlspecialchars($option); ?>" id="q<?php echo $question['id']; ?>_<?php echo array_search($option, $question['options']); ?>" style="display: none;">
                                <label for="q<?php echo $question['id']; ?>_<?php echo array_search($option, $question['options']); ?>"><?php echo htmlspecialchars($option); ?></label>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    <?php elseif ($question['type'] === 'matrix'): ?>
                        <div class="matrix-container">
                            <table class="matrix-table">
                                <thead>
                                    <tr>
                                        <th></th>
                                        <?php foreach ($question['scale'] as $scale_value): ?>
                                        <th><?php echo $scale_value; ?></th>
                                        <?php endforeach; ?>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($question['options'] as $option): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($option); ?></td>
                                        <?php foreach ($question['scale'] as $scale_value): ?>
                                        <td data-question="<?php echo $question['id']; ?>" data-option="<?php echo htmlspecialchars($option); ?>" data-value="<?php echo $scale_value; ?>">
                                            <input type="radio" name="q<?php echo $question['id']; ?>_<?php echo array_search($option, $question['options']); ?>" value="<?php echo $scale_value; ?>" style="display: none;">
                                        </td>
                                        <?php endforeach; ?>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php elseif ($question['type'] === 'open-text'): ?>
                        <textarea class="text-input" name="q<?php echo $question['id']; ?>" placeholder="<?php echo $lang === 'bg' ? 'Напишете отговора си тук...' : 'Type your answer here...'; ?>"></textarea>
                    <?php endif; ?>
                </div>
                <?php endforeach; ?>
                
                <div class="nav-buttons">
                    <button type="button" class="btn btn-secondary" id="prevBtn" style="display: none;">
                        <?php echo $current_lang['previous']; ?>
                    </button>
                    <button type="button" class="btn btn-primary" id="nextBtn">
                        <?php echo $current_lang['next']; ?>
                    </button>
                    <button type="submit" class="btn btn-primary" id="submitBtn" style="display: none;">
                        <?php echo $current_lang['submit']; ?>
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        class SurveyManager {
            constructor() {
                this.currentQuestion = 0;
                this.totalQuestions = <?php echo count($questions_data['questions']); ?>;
                this.responses = {};
                this.questions = <?php echo json_encode($questions_data['questions']); ?>;
                
                this.init();
            }
            
            init() {
                this.bindEvents();
                this.updateProgress();
                this.updateNavigation();
                
                // Ensure only the first question is visible on page load
                this.showQuestion(0);
                this.updateDebugInfo('Initial question setup complete');
            }
            
            bindEvents() {
                // Debug toggle
                const debugToggle = document.getElementById('debugToggle');
                const mobileDebug = document.getElementById('mobileDebug');
                
                if (debugToggle && mobileDebug) {
                    debugToggle.addEventListener('click', () => {
                        if (mobileDebug.style.display === 'none') {
                            mobileDebug.style.display = 'block';
                            debugToggle.textContent = '🔧 Hide Debug';
                        } else {
                            mobileDebug.style.display = 'none';
                            debugToggle.textContent = '🔧 Show Debug';
                        }
                        this.updateDebugInfo();
                    });
                }
                
                // Navigation buttons
                const nextBtn = document.getElementById('nextBtn');
                const prevBtn = document.getElementById('prevBtn');
                
                if (nextBtn) {
                    nextBtn.addEventListener('click', (e) => {
                        e.preventDefault();
                        e.stopPropagation();
                        this.updateDebugInfo('Next button clicked');
                        this.nextQuestion();
                    });
                }
                
                if (prevBtn) {
                    prevBtn.addEventListener('click', (e) => {
                        e.preventDefault();
                        e.stopPropagation();
                        this.updateDebugInfo('Previous button clicked');
                        this.previousQuestion();
                    });
                }
                
                // Form submission
                const form = document.getElementById('surveyForm');
                if (form) {
                    form.addEventListener('submit', (e) => {
                        this.handleSubmit(e);
                    });
                }
                
                // Option selection
                document.querySelectorAll('.option').forEach(option => {
                    option.addEventListener('click', (e) => this.handleOptionClick(e));
                });
                
                // Matrix selection
                document.querySelectorAll('.matrix-table td').forEach(cell => {
                    if (cell.dataset.value) {
                        cell.addEventListener('click', (e) => this.handleMatrixClick(e));
                    }
                });
                
                // Text input
                document.querySelectorAll('.text-input').forEach(input => {
                    input.addEventListener('input', (e) => this.handleTextInput(e));
                });
                
                // Initial debug info
                this.updateDebugInfo('Page loaded');
            }
            
            handleOptionClick(e) {
                const option = e.currentTarget;
                const questionId = this.getQuestionId(option);
                const value = option.dataset.value;
                
                // Handle single choice
                const radio = option.querySelector('input[type="radio"]');
                if (radio) {
                    // Remove selected class from all options in this question
                    const questionContainer = option.closest('.question-container');
                    questionContainer.querySelectorAll('.option').forEach(opt => opt.classList.remove('selected'));
                    
                    // Add selected class to clicked option
                    option.classList.add('selected');
                    radio.checked = true;
                    
                    this.responses[questionId] = value;
                }
                
                // Handle multi choice
                const checkbox = option.querySelector('input[type="checkbox"]');
                if (checkbox) {
                    if (checkbox.checked) {
                        option.classList.remove('selected');
                        checkbox.checked = false;
                        this.removeFromMultiResponse(questionId, value);
                    } else {
                        option.classList.add('selected');
                        checkbox.checked = true;
                        this.addToMultiResponse(questionId, value);
                    }
                }
                
                this.updateNavigation();
            }
            
            handleMatrixClick(e) {
                const cell = e.currentTarget;
                const questionId = cell.dataset.question;
                const option = cell.dataset.option;
                const value = cell.dataset.value;
                
                // Remove selected class from all cells in this row
                const row = cell.closest('tr');
                row.querySelectorAll('td').forEach(td => td.classList.remove('selected'));
                
                // Add selected class to clicked cell
                cell.classList.add('selected');
                
                // Check the radio button
                const radio = cell.querySelector('input[type="radio"]');
                radio.checked = true;
                
                // Store response
                if (!this.responses[questionId]) {
                    this.responses[questionId] = {};
                }
                this.responses[questionId][option] = value;
                
                this.updateNavigation();
            }
            
            handleTextInput(e) {
                const input = e.currentTarget;
                const questionId = this.getQuestionId(input);
                const value = input.value.trim();
                
                if (value) {
                    this.responses[questionId] = value;
                } else {
                    delete this.responses[questionId];
                }
                
                this.updateNavigation();
            }
            
            getQuestionId(element) {
                const questionContainer = element.closest('.question-container');
                return questionContainer.dataset.question;
            }
            
            addToMultiResponse(questionId, value) {
                if (!this.responses[questionId]) {
                    this.responses[questionId] = [];
                }
                if (!this.responses[questionId].includes(value)) {
                    this.responses[questionId].push(value);
                }
            }
            
            removeFromMultiResponse(questionId, value) {
                if (this.responses[questionId] && Array.isArray(this.responses[questionId])) {
                    this.responses[questionId] = this.responses[questionId].filter(item => item !== value);
                    if (this.responses[questionId].length === 0) {
                        delete this.responses[questionId];
                    }
                }
            }
            
            nextQuestion() {
                this.updateDebugInfo('Next button clicked. Current: ' + this.currentQuestion + ', Total: ' + this.totalQuestions);
                if (this.currentQuestion < this.totalQuestions - 1) {
                    this.currentQuestion++;
                    this.updateDebugInfo('Moving to question: ' + this.currentQuestion);
                    this.showQuestion(this.currentQuestion);
                    this.updateProgress();
                    this.updateNavigation();
                } else {
                    this.updateDebugInfo('Already at last question');
                }
            }
            
            previousQuestion() {
                if (this.currentQuestion > 0) {
                    this.currentQuestion--;
                    this.showQuestion(this.currentQuestion);
                    this.updateProgress();
                    this.updateNavigation();
                }
            }
            
            showQuestion(index) {
                this.updateDebugInfo('Showing question: ' + index);
                
                const containers = document.querySelectorAll('.question-container');
                this.updateDebugInfo('Total containers: ' + containers.length);
                
                // First, hide all containers
                containers.forEach((container, i) => {
                    container.classList.remove('active');
                    container.style.display = 'none';
                    container.style.opacity = '0';
                    container.style.transform = 'translateY(20px)';
                });
                
                // Then show the active one
                if (containers[index]) {
                    containers[index].classList.add('active');
                    containers[index].style.display = 'block';
                    
                    // Force mobile repaint
                    setTimeout(() => {
                        containers[index].style.opacity = '1';
                        containers[index].style.transform = 'translateY(0)';
                        this.updateDebugInfo('Question ' + index + ' should be visible now');
                    }, 50);
                }
                
                this.updateDebugInfo('Question display updated');
            }
            
            updateProgress() {
                const progress = ((this.currentQuestion + 1) / this.totalQuestions) * 100;
                document.getElementById('progressFill').style.width = progress + '%';
                document.getElementById('progressText').textContent = 
                    '<?php echo $current_lang['question']; ?> ' + (this.currentQuestion + 1) + 
                    ' <?php echo $current_lang['of']; ?> ' + this.totalQuestions;
            }
            
            updateDebugInfo(action = '') {
                const debugCurrent = document.getElementById('debugCurrent');
                const debugTotal = document.getElementById('debugTotal');
                const debugVisible = document.getElementById('debugVisible');
                const debugActive = document.getElementById('debugActive');
                const debugLastAction = document.getElementById('debugLastAction');
                
                if (debugCurrent) debugCurrent.textContent = 'Current: ' + this.currentQuestion;
                if (debugTotal) debugTotal.textContent = 'Total: ' + this.totalQuestions;
                
                const visibleQuestions = document.querySelectorAll('.question-container[style*="display: block"], .question-container.active').length;
                if (debugVisible) debugVisible.textContent = 'Visible Questions: ' + visibleQuestions;
                
                const activeQuestion = document.querySelector('.question-container.active');
                if (debugActive) debugActive.textContent = 'Active Question: ' + (activeQuestion ? 'Found' : 'None');
                
                if (debugLastAction && action) debugLastAction.textContent = 'Last Action: ' + action;
            }
            
            updateNavigation() {
                const prevBtn = document.getElementById('prevBtn');
                const nextBtn = document.getElementById('nextBtn');
                const submitBtn = document.getElementById('submitBtn');
                
                // Show/hide previous button
                prevBtn.style.display = this.currentQuestion > 0 ? 'inline-block' : 'none';
                
                // Show/hide next/submit buttons
                if (this.currentQuestion === this.totalQuestions - 1) {
                    nextBtn.style.display = 'none';
                    submitBtn.style.display = 'inline-block';
                    
                    // Only disable submit button if current question is not answered
                    const currentQuestionId = this.questions[this.currentQuestion].id;
                    const isAnswered = this.isQuestionAnswered(currentQuestionId);
                    submitBtn.disabled = !isAnswered;
                } else {
                    nextBtn.style.display = 'inline-block';
                    submitBtn.style.display = 'none';
                    
                    // Always enable next button - users can navigate freely
                    nextBtn.disabled = false;
                }
            }
            
            isQuestionAnswered(questionId) {
                const question = this.questions.find(q => q.id == questionId);
                
                if (question.type === 'open-text') {
                    return true; // Open text is optional
                }
                
                if (question.type === 'multi-choice') {
                    return this.responses[questionId] && this.responses[questionId].length > 0;
                }
                
                if (question.type === 'matrix') {
                    return this.responses[questionId] && Object.keys(this.responses[questionId]).length === question.options.length;
                }
                
                return this.responses[questionId] !== undefined;
            }
            
            handleSubmit(e) {
                e.preventDefault();
                
                // Update hidden input with responses
                document.getElementById('responsesInput').value = JSON.stringify(this.responses);
                
                // Submit form
                document.getElementById('surveyForm').submit();
            }
        }
        
        // Initialize survey when page loads
        document.addEventListener('DOMContentLoaded', function() {
            new SurveyManager();
        });
    </script>
</body>
</html>
