<?php
session_start();

$lang = $_GET['lang'] ?? 'en';

// Language translations
$translations = [
    'en' => [
        'title' => 'Survey Response Analysis',
        'no_responses' => 'No responses found yet.',
        'total_responses' => 'Total Responses',
        'buyer_responses' => 'Buyer Responses',
        'seller_responses' => 'Seller Responses',
        'language_distribution' => 'Language Distribution',
        'user_paths' => 'User Paths',
        'response_timeline' => 'Response Timeline',
        'view_details' => 'View Details',
        'back_to_start' => 'Back to Start',
        'date' => 'Date',
        'language' => 'Language',
        'path' => 'Path',
        'type' => 'Type',
        'responses' => 'Responses',
        'download_csv' => 'Download CSV',
        'filter_by_date' => 'Filter by Date',
        'all_dates' => 'All Dates'
    ],
    'bg' => [
        'title' => 'Анализ на отговорите от проучването',
        'no_responses' => 'Все още няма намерени отговори.',
        'total_responses' => 'Общо отговори',
        'buyer_responses' => 'Отговори от купувачи',
        'seller_responses' => 'Отговори от продавачи',
        'language_distribution' => 'Разпределение по език',
        'user_paths' => 'Пътища на потребителите',
        'response_timeline' => 'Времева линия на отговорите',
        'view_details' => 'Преглед на детайли',
        'back_to_start' => 'Обратно към началото',
        'date' => 'Дата',
        'language' => 'Език',
        'path' => 'Път',
        'type' => 'Тип',
        'responses' => 'Отговори',
        'download_csv' => 'Изтегли CSV',
        'filter_by_date' => 'Филтрирай по дата',
        'all_dates' => 'Всички дати'
    ]
];

$current_lang = $translations[$lang] ?? $translations['en'];

// Load responses
$responses = [];
$responses_dir = 'responses/';
if (is_dir($responses_dir)) {
    $files = glob($responses_dir . '*.json');
    foreach ($files as $file) {
        $data = json_decode(file_get_contents($file), true);
        if ($data) {
            $responses = array_merge($responses, $data);
        }
    }
}

// Calculate statistics
$stats = [
    'total' => count($responses),
    'buyer' => 0,
    'seller' => 0,
    'languages' => [],
    'paths' => [],
    'dates' => []
];

foreach ($responses as $response) {
    $stats['buyer'] += ($response['survey_type'] === 'buyer') ? 1 : 0;
    $stats['seller'] += ($response['survey_type'] === 'seller') ? 1 : 0;
    
    $stats['languages'][$response['language']] = ($stats['languages'][$response['language']] ?? 0) + 1;
    $stats['paths'][$response['user_path']] = ($stats['paths'][$response['user_path']] ?? 0) + 1;
    
    $date = date('Y-m-d', strtotime($response['timestamp']));
    $stats['dates'][$date] = ($stats['dates'][$date] ?? 0) + 1;
}

// Handle CSV download
if (isset($_GET['download']) && $_GET['download'] === 'csv') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="survey_responses_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // CSV headers
    fputcsv($output, ['Timestamp', 'Language', 'User Path', 'Survey Type', 'Question ID', 'Response']);
    
    foreach ($responses as $response) {
        $responses_data = is_string($response['responses']) ? json_decode($response['responses'], true) : $response['responses'];
        
        if ($responses_data) {
            foreach ($responses_data as $question_id => $answer) {
                if (is_array($answer)) {
                    $answer = implode(', ', $answer);
                }
                fputcsv($output, [
                    $response['timestamp'],
                    $response['language'],
                    $response['user_path'],
                    $response['survey_type'],
                    $question_id,
                    $answer
                ]);
            }
        }
    }
    
    fclose($output);
    exit;
}
?>

<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $current_lang['title']; ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 12px;
            text-align: center;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
        .stat-number {
            font-size: 2rem;
            font-weight: bold;
            color: #667eea;
            margin-bottom: 10px;
        }
        .stat-label {
            color: #666;
            font-size: 0.9rem;
        }
        .responses-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
        .responses-table th,
        .responses-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #e9ecef;
        }
        .responses-table th {
            background: #f8f9fa;
            font-weight: 600;
        }
        .responses-table tr:hover {
            background: #f8f9ff;
        }
        .filter-controls {
            margin-bottom: 20px;
            display: flex;
            gap: 15px;
            align-items: center;
            flex-wrap: wrap;
        }
        .filter-controls select,
        .filter-controls input {
            padding: 8px 12px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 0.9rem;
        }
        .filter-controls select:focus,
        .filter-controls input:focus {
            outline: none;
            border-color: #667eea;
        }
        .download-btn {
            background: #28a745;
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .download-btn:hover {
            background: #218838;
            transform: translateY(-2px);
        }
        .no-data {
            text-align: center;
            padding: 40px;
            color: #666;
            font-style: italic;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="survey-container">
            <div class="survey-header">
                <h1 class="survey-title"><?php echo $current_lang['title']; ?></h1>
            </div>
            
            <?php if (empty($responses)): ?>
            <div class="no-data">
                <p><?php echo $current_lang['no_responses']; ?></p>
            </div>
            <?php else: ?>
            
            <!-- Statistics Cards -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-number"><?php echo $stats['total']; ?></div>
                    <div class="stat-label"><?php echo $current_lang['total_responses']; ?></div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php echo $stats['buyer']; ?></div>
                    <div class="stat-label"><?php echo $current_lang['buyer_responses']; ?></div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php echo $stats['seller']; ?></div>
                    <div class="stat-label"><?php echo $current_lang['seller_responses']; ?></div>
                </div>
            </div>
            
            <!-- Language Distribution -->
            <div class="stat-card">
                <h3><?php echo $current_lang['language_distribution']; ?></h3>
                <?php foreach ($stats['languages'] as $lang_code => $count): ?>
                <p><strong><?php echo strtoupper($lang_code); ?>:</strong> <?php echo $count; ?> (<?php echo round(($count / $stats['total']) * 100, 1); ?>%)</p>
                <?php endforeach; ?>
            </div>
            
            <!-- User Paths -->
            <div class="stat-card">
                <h3><?php echo $current_lang['user_paths']; ?></h3>
                <?php foreach ($stats['paths'] as $path => $count): ?>
                <p><strong><?php echo ucfirst($path); ?>:</strong> <?php echo $count; ?> (<?php echo round(($count / $stats['total']) * 100, 1); ?>%)</p>
                <?php endforeach; ?>
            </div>
            
            <!-- Filter Controls -->
            <div class="filter-controls">
                <label><?php echo $current_lang['filter_by_date']; ?>:</label>
                <select id="dateFilter">
                    <option value=""><?php echo $current_lang['all_dates']; ?></option>
                    <?php foreach (array_keys($stats['dates']) as $date): ?>
                    <option value="<?php echo $date; ?>"><?php echo $date; ?> (<?php echo $stats['dates'][$date]; ?> responses)</option>
                    <?php endforeach; ?>
                </select>
                
                <a href="?download=csv&lang=<?php echo $lang; ?>" class="download-btn">
                    <?php echo $current_lang['download_csv']; ?>
                </a>
            </div>
            
            <!-- Responses Table -->
            <table class="responses-table">
                <thead>
                    <tr>
                        <th><?php echo $current_lang['date']; ?></th>
                        <th><?php echo $current_lang['language']; ?></th>
                        <th><?php echo $current_lang['path']; ?></th>
                        <th><?php echo $current_lang['type']; ?></th>
                        <th><?php echo $current_lang['responses']; ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($responses as $response): ?>
                    <tr data-date="<?php echo date('Y-m-d', strtotime($response['timestamp'])); ?>">
                        <td><?php echo date('M j, Y H:i', strtotime($response['timestamp'])); ?></td>
                        <td><?php echo strtoupper($response['language']); ?></td>
                        <td><?php echo ucfirst($response['user_path']); ?></td>
                        <td><?php echo ucfirst($response['survey_type']); ?></td>
                        <td>
                            <?php 
                            $responses_data = is_string($response['responses']) ? json_decode($response['responses'], true) : $response['responses'];
                            if ($responses_data) {
                                echo '<details><summary>' . count($responses_data) . ' answers</summary>';
                                echo '<div style="margin-top: 10px;">';
                                foreach ($responses_data as $question_id => $answer) {
                                    if (is_array($answer)) {
                                        $answer = implode(', ', $answer);
                                    }
                                    echo '<div><strong>Q' . $question_id . ':</strong> ' . htmlspecialchars($answer) . '</div>';
                                }
                                echo '</div></details>';
                            } else {
                                echo 'No responses';
                            }
                            ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
            <?php endif; ?>
            
            <div class="nav-buttons" style="justify-content: center; margin-top: 40px;">
                <a href="index.php" class="btn btn-secondary">
                    <?php echo $current_lang['back_to_start']; ?>
                </a>
            </div>
        </div>
    </div>

    <script>
        // Date filter functionality
        document.getElementById('dateFilter').addEventListener('change', function() {
            const selectedDate = this.value;
            const rows = document.querySelectorAll('.responses-table tbody tr');
            
            rows.forEach(row => {
                const rowDate = row.dataset.date;
                if (!selectedDate || rowDate === selectedDate) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
    </script>
</body>
</html>
